using System;
using System.Xml;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// Factory class analyzing supplied VistALink fault message and creating 
	/// appropriate VistALink fault exception.
	/// </summary>
	public class VistALinkFaultExceptionFactory
	{
		/// <summary>
		/// Default parameterless constructor added for extendibility.
		/// The only public method of this class is made instance virtual method, so
		/// it may be overriden by derived classes in the future.
		/// </summary>
		public VistALinkFaultExceptionFactory() {}

		/// <summary>
		/// This method will analyze supplied VistALink fault message and create 
		/// appropriate exception based on its properties.
		/// It will create the most specific exception possible and return general VistALinkFaultException
		/// if fault info is not recognized.
		/// </summary>
		/// <param name="faultMessage">VistALink fault message received from remote server.</param>
		/// <returns>
		///		New instance of VistALinkFaultException or one of its children created from fault message.		
		///	</returns>
		public virtual VistALinkFaultException CreateVistALinkExceptionFromFaultMessage( BaseFaultMessage faultMessage )
		{
			if( faultMessage == null )
				throw( new ArgumentNullException( "faultMessage" ) );

			if( faultMessage is VistALinkFaultMessage )
			{
				if( faultMessage.FaultInfo.ErrorCode == "181003" )
					return new NoJobSlotsAvailableException( faultMessage.FaultInfo );

				if( faultMessage.FaultInfo.ErrorCode == "181004" )
					return new LoginsDisabledException( faultMessage.FaultInfo );

				// General system fault exception
				return new SystemFaultException( faultMessage.FaultInfo );			
			}

			if( faultMessage is SecurityFaultMessage )
			{
				if( faultMessage.FaultInfo.ErrorCode == "183003" )
					return new AlreadyLoggedOnException( faultMessage.FaultInfo );

				// General security fault exception
				return new SecurityFaultException( faultMessage.FaultInfo );
			}
		
			if( faultMessage is RpcFaultMessage )
			{
				if( faultMessage.FaultInfo.ErrorCode == "182007" )
					return new RpcTimeoutException( faultMessage.FaultInfo );

				// General RPC fault exception
				return new RpcFaultException( faultMessage.FaultInfo );				
			}

			// General VistALink fault exception
			return new VistALinkFaultException( faultMessage.FaultInfo );
		}
	}
}
